/*
    Copyright (C) 2010  C-LIS CO., LTD.

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
 */

package jp.co.c_lis.ccl.medicinesearch.android.net;

import java.io.BufferedInputStream;
import java.io.IOException;
import java.net.URLEncoder;
import java.util.LinkedList;
import java.util.List;

import jp.co.c_lis.ccl.medicinesearch.android.entity.Medicine;
import jp.co.c_lis.ccl.medicinesearch.android.entity.MedicineDetail;
import jp.co.c_lis.ccl.medicinesearch.android.entity.MedicineDetail2;

import org.apache.http.HttpResponse;
import org.apache.http.client.ClientProtocolException;
import org.apache.http.client.methods.HttpGet;
import org.apache.http.impl.client.DefaultHttpClient;
import org.apache.http.util.ByteArrayBuffer;
import org.xml.sax.Attributes;
import org.xml.sax.ContentHandler;
import org.xml.sax.SAXException;
import org.xml.sax.XMLReader;
import org.xml.sax.helpers.DefaultHandler;

import android.graphics.Bitmap;
import android.graphics.BitmapFactory;
import android.text.Editable;
import android.text.Html;
import android.util.Log;

/**
 * Yahoo! Japan ヘルスケア お薬検索 情報取得クラス
 * 
 * @since Dec 28, 2009
 * @version 2010011001
 */
public class YahooHealthServerInterface {
    private static final boolean DEBUG_FLAG = false;
    private static final String LOG_TAG = "YahooHealthServerInterface";

    /**
     * コンストラクタ
     */
    private YahooHealthServerInterface() {}

    // 基本URL
    private static final String BASE_URL = "http://health.yahoo.co.jp/";

    /**
     * 薬のリストと、ページ情報を保存するクラス
     */
    public static class MedicineList {

        public List<Medicine> list = new LinkedList<Medicine>();
        int pages = -1;
        int nowPage = -1;

        public boolean hasNexePage() {
            return (pages > nowPage);
        }

    }

    /**
     * 検索用Url取得
     * 
     * @param keyword
     * @param page
     * @return
     */
    private static final String generateSearchUrlString(String keyword, int page) {
        String encodedKeyword = URLEncoder.encode(keyword);
        String result = BASE_URL + "search/?p=" + encodedKeyword + "&type=medicine";
        if (page > 1) result += "&b=" + 10 * (page - 1) + "&pageFlg=1";
        return result;
    }

    /**
     * 薬のリストを取得（１ページ目）
     * 
     * @param keyword
     * @return
     * @throws IOException
     */
    public static MedicineList getMedicineList(String keyword) throws IOException {
        return getMedicineList(keyword, 1);
    }

    /**
     * HTTPアクセスして、HTMLを取得
     * 
     * @param urlString
     *            GETパラメータを含むURL
     * @return
     * @throws IOException
     */
    private static String httpAccess(String urlString) throws IOException {

        String result = null;

        HttpGet entity = new HttpGet(urlString);

        BufferedInputStream reader = null;
        try {
            DefaultHttpClient request = new DefaultHttpClient();
            HttpResponse res = request.execute(entity);

            reader = new BufferedInputStream(res.getEntity().getContent());

            ByteArrayBuffer bab = new ByteArrayBuffer(512);
            byte[] buff = new byte[512];
            int len = 0;
            while ((len = reader.read(buff)) > 0) {
                bab.append(buff, 0, len);
            }

            result = new String(bab.toByteArray(), "EUC_JP");

            // オブジェクトの解放
            bab = null;

        } catch (ClientProtocolException e) {
            Log.e(LOG_TAG, e.getMessage());
        } catch (IllegalStateException e) {
            Log.e(LOG_TAG, e.getMessage());
        } finally {
            if (reader != null) {
                reader.close();
            }
        }

        // 文字列に変換して返却
        return result;

    }

    /**
     * 薬のリストを取得
     * 
     * @param keyword
     * @param page
     * @return
     * @throws IOException
     */
    public static MedicineList getMedicineList(String keyword, int page) throws IOException {
        MedicineList result = null;

        String html = httpAccess(generateSearchUrlString(keyword, page));
        result = getMedicineListFromHtml(html);

        return result;
    }

    /**
     * HTMLを解析して、薬のリストを取得する
     * 
     * @param html
     * @return Medicineオブジェクトが格納されたリスト
     */
    public static final MedicineList getMedicineListFromHtml(String html) {

        html = html.replace("<br>", "\n");

        final MedicineList result = new MedicineList();
        final ContentHandler medicineListHandler = new DefaultHandler() {

            private boolean divTagListTabOpened = false;
            private boolean pagesReady = false;
            private boolean nowPageReady = false;

            private boolean trTagOpened = false;
            private boolean tdTagOpened = false;
            private boolean nameTagOpened = false;

            private boolean nameReady = false;
            private boolean companyReady = false;
            private boolean shapeReady = false;
            private boolean effectReady = false;
            private boolean medicineCodeReady = false;

            private Medicine medicine = null;

            @Override
            public void startElement(String uri, String localName, String qName, Attributes atts)
                    throws SAXException {
                String clazz = atts.getValue("class");

                if (localName.equals("div")) {
                    if (clazz != null && clazz.equals("list-tab")) {
                        divTagListTabOpened = true;
                    }
                } else if (divTagListTabOpened && localName.equals("strong")) {
                    if (pagesReady == false) {
                        pagesReady = true;
                    } else {
                        nowPageReady = true;
                    }
                } else if (localName.equals("tr")) {
                    trTagOpened = true;
                    if (medicine == null) medicine = new Medicine();

                } else if (nameTagOpened && localName.equals("img")) {
                    medicine.setType(atts.getValue("alt"));

                } else if (nameTagOpened && localName.equals("a")) {
                    nameReady = true;
                    medicine.setLink(atts.getValue("href"));

                } else if (trTagOpened && localName.equals("td")) {
                    tdTagOpened = true;
                    if (clazz == null) return;

                    if (clazz.equals("name")) {
                        nameTagOpened = true;
                    } else if (clazz.equals("company")) {
                        companyReady = true;
                    } else if (clazz.equals("shape")) {
                        shapeReady = true;
                    } else if (clazz.equals("effect")) {
                        effectReady = true;
                    } else if (clazz.equals("medicineCode")) {
                        medicineCodeReady = true;
                    }
                }
            }

            @Override
            public void endElement(String uri, String localName, String qName) throws SAXException {
                String value = sb.toString();
                sb.delete(0, sb.length());

                if (localName.equals("div") && divTagListTabOpened) {
                    divTagListTabOpened = false;
                    pagesReady = false;
                    nowPageReady = false;
                } else if (localName.equals("strong") && divTagListTabOpened) {
                    if (pagesReady && result.pages == -1) {
                        result.pages = Integer.parseInt(value);
                    } else if (nowPageReady && result.nowPage == -1) {
                        result.nowPage = Integer.parseInt(value);
                    }
                } else if (localName.equals("br") && tdTagOpened) {
                    sb.append("\n");
                } else if (localName.equals("tr") && trTagOpened) {
                    trTagOpened = false;
                } else if (localName.equals("a")) {
                    if (nameReady) {
                        medicine.setName(value);
                        nameReady = false;
                    }
                } else if (localName.equals("td")) {
                    tdTagOpened = false;
                    if (nameTagOpened) {
                        nameTagOpened = false;

                    } else if (companyReady) {
                        medicine.company = value;
                        companyReady = false;

                    } else if (shapeReady) {
                        medicine.shape = value;
                        shapeReady = false;

                    } else if (effectReady) {
                        medicine.effect = value;
                        effectReady = false;

                    } else if (medicineCodeReady) {
                        medicine.setMedicineCode(value);
                        medicineCodeReady = false;

                        if (DEBUG_FLAG) Log.d(LOG_TAG, medicine.toString());

                        // リストへ追加
                        result.list.add(medicine);
                        medicine = null;
                    }
                }
            }

            private StringBuilder sb = new StringBuilder();

            @Override
            public void characters(char[] ch, int start, int length) throws SAXException {
                if (tdTagOpened || (pagesReady && result.pages == -1)
                        || (nowPageReady && result.nowPage == -1)) {
                    sb.append(ch, start, length);
                }
            }
        };

        Html.ImageGetter imageGetter = null;
        Html.TagHandler tagHandler = new Html.TagHandler() {
            @Override
            public void handleTag(boolean opening, String tag, Editable output, XMLReader xmlReader) {
                if (tag.equals("table") && opening) {
                    xmlReader.setContentHandler(medicineListHandler);
                }
            }
        };

        Html.fromHtml(html.toString(), imageGetter, tagHandler);

        return result;
    }

    /**
     * 詳細取得Url取得
     * 
     * @param medicine
     * @return
     */
    private static final String generateDetailUrlString(Medicine medicine) {
        return BASE_URL + "hospital/medicine/detail/" + medicine.getType() + "/" + medicine.id
                + "/";
    }

    /**
     * 薬の詳細情報を取得
     * 
     * @param medicine
     * @return
     * @throws IOException
     */
    public static Medicine getMedicineDetail(Medicine medicine) throws IOException {

        Medicine result = new MedicineDetail(medicine);

        String html = httpAccess(generateDetailUrlString(medicine));
        result = getMedicineDetailFromHtml(html);
        result.setMedicine(medicine);

        return result;
    }

    /**
     * 
     * @param html
     * @return
     */
    private static final MedicineDetail parseHtmlAsPattern1(String html) {

        final MedicineDetail result = new MedicineDetail();

        final ContentHandler medicineDetailHandler = new DefaultHandler() {

            private boolean divTagMds1Opened = false;
            private boolean divTagRxdataOpened = false;
            private boolean dlTagOpened = false;
            private boolean dtTagOpened = false;
            private boolean ddTagOpened = false;

            private boolean nameReady = false;
            private boolean companyReady = false;
            private boolean typeReady = false;
            private boolean generalNameReady = false;
            private boolean codeReady = false;
            private boolean shapeReady = false;
            private boolean unitReady = false;
            private boolean priceUnitReady = false;
            private boolean effectReady = false;
            private boolean mainEffectReady = false;
            private boolean sideEffectReady = false;
            private boolean usageReady = false;
            private boolean picturesReady = false;
            private boolean sourceReady = false;

            private void setReadyFlg(String alt) {
                if (alt == null) return;

                if (alt.equals("製品名")) {
                    nameReady = true;
                } else if (alt.equals("製薬会社名")) {
                    companyReady = true;
                } else if (alt.equals("分類")) {
                    typeReady = true;
                } else if (alt.equals("一般名")) {
                    generalNameReady = true;
                } else if (alt.equals("識別コード")) {
                    codeReady = true;
                } else if (alt.equals("形状")) {
                    shapeReady = true;
                } else if (alt.equals("規格単位")) {
                    unitReady = true;
                } else if (alt.equals("薬価")) {
                    priceUnitReady = true;
                } else if (alt.equals("種別")) {
                    effectReady = true;
                } else if (alt.equals("主な作用")) {
                    mainEffectReady = true;
                } else if (alt.equals("副作用など")) {
                    sideEffectReady = true;
                } else if (alt.equals("用い方と注意")) {
                    usageReady = true;
                } else if (alt.equals("製品写真")) {
                    picturesReady = true;
                }
            }

            @Override
            public void startElement(String uri, String localName, String qName, Attributes atts)
                    throws SAXException {
                String clazz = atts.getValue("class");
                String id = atts.getValue("id");

                if (localName.equals("div") && divTagRxdataOpened == false) {

                    // if (DEBUG_FLAG) Log.d(LOG_TAG, "id = " + id);
                    if (divTagMds1Opened == false && clazz != null && clazz.equals("mds1")) {
                        divTagMds1Opened = true;
                    } else if (divTagMds1Opened && id != null && id.equals("rxdata")) {
                        divTagRxdataOpened = true;
                    }

                } else if (divTagRxdataOpened) {

                    if (localName.equals("dl")) {
                        dlTagOpened = true;
                    } else if (dlTagOpened && localName.equals("dt")) {
                        dtTagOpened = true;
                    } else if (dlTagOpened && localName.equals("dd")) {
                        ddTagOpened = true;
                    } else if (dtTagOpened && localName.equals("img")) {
                        setReadyFlg(atts.getValue("alt"));
                    } else if (picturesReady && localName.equals("img")) {
                        result.imageList.add(atts.getValue("src"));
                    } else if (localName.equals("p")) {
                        if (clazz != null && clazz.equals("note")) {
                            sourceReady = true;
                        }
                    }
                }
            }

            @Override
            public void endElement(String uri, String localName, String qName) throws SAXException {
                String value = sb.toString();
                sb.delete(0, sb.length());

                if (localName.equals("br") && ddTagOpened) {
                    sb.append("\n");
                } else if (localName.equals("dl")) {
                    dlTagOpened = false;
                } else if (dlTagOpened && localName.equals("dt")) {
                    dtTagOpened = false;
                } else if (dlTagOpened && localName.equals("dd")) {
                    ddTagOpened = false;
                    if (nameReady) {
                        result.setName(value);
                        nameReady = false;
                    } else if (companyReady) {
                        result.company = value;
                        companyReady = false;
                    } else if (typeReady) {
                        result.setType(value);
                        typeReady = false;
                    } else if (generalNameReady) {
                        result.generalName = value;
                        generalNameReady = false;
                    } else if (codeReady) {
                        result.setMedicineCode(value);
                        codeReady = false;
                    } else if (shapeReady) {
                        result.shape = value;
                        shapeReady = false;
                    } else if (unitReady) {
                        result.unit = value;
                        unitReady = false;
                    } else if (priceUnitReady) {
                        result.priceUnit = Float.parseFloat(value);
                        priceUnitReady = false;
                    } else if (effectReady) {
                        result.effect = value;
                        effectReady = false;
                    } else if (mainEffectReady) {
                        result.mainEffect = value;
                        mainEffectReady = false;
                    } else if (sideEffectReady) {
                        result.sideEffect = value;
                        sideEffectReady = false;
                    } else if (usageReady) {
                        result.usage = value;
                        usageReady = false;
                    } else if (picturesReady) {
                        picturesReady = false;
                    }
                } else if (sourceReady && localName.equals("p")) {
                    result.source = value;
                    sourceReady = false;
                }
            }

            private StringBuilder sb = new StringBuilder();

            @Override
            public void characters(char[] ch, int start, int length) throws SAXException {
                if (ddTagOpened || sourceReady) {
                    sb.append(ch, start, length);
                }
            }
        };

        Html.ImageGetter imageGetter = null;
        Html.TagHandler tagHandler = new Html.TagHandler() {
            @Override
            public void handleTag(boolean opening, String tag, Editable output, XMLReader xmlReader) {
                if (tag.equals("body") && opening) {
                    xmlReader.setContentHandler(medicineDetailHandler);
                }
            }
        };

        Html.fromHtml(html.toString(), imageGetter, tagHandler);

        return result;
    }

    /**
     * 
     * @param html
     * @return
     */
    private static final Medicine parseHtmlAsPattern2(String html) {

        final MedicineDetail2 result = new MedicineDetail2();

        final ContentHandler medicineDetailHandler = new DefaultHandler() {

            private boolean divTagMds1Opened = false;
            private boolean divTagDataLeftOpened = false;
            private boolean divTagOctDataOpened = false;

            private boolean dlTagOpened = false;
            private boolean dtTagOpened = false;
            private boolean ddTagOpened = false;

            private boolean namePreready = false;
            private boolean nameReady = false;

            private boolean companyReady = false;
            private boolean typeReady = false;
            private boolean especiallyReady = false;

            private boolean elementReady = false;

            private boolean shapeReady = false;
            private boolean effectReady = false;
            private boolean mainEffectReady = false;
            private boolean usageReady = false;

            private boolean parcelAndPriceReady = false;
            private boolean carefulReady = false;
            private boolean riskTypeReady = false;

            private boolean sourceReady = false;

            private void setLeftReadyFlg(String alt) {
                if (alt == null) return;

                if (alt.equals("製品名")) {
                    namePreready = true;
                } else if (alt.equals("製薬会社名")) {
                    companyReady = true;
                } else if (alt.equals("分類")) {
                    typeReady = true;
                } else if (alt.equals("特徴")) {
                    especiallyReady = true;
                }
            }

            private void setOctDataReadyFlg(String alt) {
                if (alt == null) return;

                if (alt.equals("成分")) {
                    elementReady = true;
                } else if (alt.equals("効能")) {
                    effectReady = true;
                } else if (alt.equals("主な作用")) {
                    mainEffectReady = true;
                } else if (alt.equals("用法・用量")) {
                    usageReady = true;
                } else if (alt.equals("形状")) {
                    shapeReady = true;
                } else if (alt.equals("包装単位と希望小売価格")) {
                    parcelAndPriceReady = true;
                } else if (alt.equals("製品特有の注意")) {
                    carefulReady = true;
                } else if (alt.equals("リスク分類ほか")) {
                    riskTypeReady = true;
                }
            }

            @Override
            public void startElement(String uri, String localName, String qName, Attributes atts)
                    throws SAXException {
                String clazz = atts.getValue("class");
                String id = atts.getValue("id");

                if (localName.equals("div")) {

                    if (divTagMds1Opened == false && clazz != null && clazz.equals("mds1")) {
                        divTagMds1Opened = true;
                    } else if (divTagMds1Opened && id != null && id.equals("dataleft")) {
                        divTagDataLeftOpened = true;
                    } else if (divTagMds1Opened && id != null && id.equals("otcdata-lower")) {
                        divTagOctDataOpened = true;
                    }
                }
                if (divTagDataLeftOpened && divTagOctDataOpened == false) {

                    if (localName.equals("dl")) {
                        dlTagOpened = true;
                    } else if (dlTagOpened && localName.equals("dt")) {
                        dtTagOpened = true;
                    } else if (dlTagOpened && localName.equals("dd")) {
                        ddTagOpened = true;
                    } else if (dtTagOpened && localName.equals("img")) {
                        setLeftReadyFlg(atts.getValue("alt"));
                    } else if (localName.equals("p")) {
                        if (ddTagOpened && namePreready) {
                            nameReady = true;
                            if (DEBUG_FLAG) Log.d(LOG_TAG, "nameReady");
                        }
                    }
                } else if (divTagOctDataOpened) {

                    if (localName.equals("dl")) {
                        dlTagOpened = true;
                    } else if (dlTagOpened && localName.equals("dt")) {
                        dtTagOpened = true;
                    } else if (dlTagOpened && localName.equals("dd")) {
                        ddTagOpened = true;
                    } else if (dtTagOpened && localName.equals("img")) {
                        setOctDataReadyFlg(atts.getValue("alt"));
                    } else if (localName.equals("p")) {
                        if (clazz != null && clazz.equals("txsleft")) {
                            sourceReady = true;
                        }
                    }
                }
            }

            @Override
            public void endElement(String uri, String localName, String qName) throws SAXException {
                String value = sb.toString();
                sb.delete(0, sb.length());

                if (localName.equals("br") && ddTagOpened) {
                    sb.append("\n");
                } else if (localName.equals("dl")) {
                    dlTagOpened = false;
                } else if (dlTagOpened && localName.equals("dt")) {
                    dtTagOpened = false;
                } else if (nameReady && localName.equals("p")) {
                    result.setName(value.trim());
                    nameReady = false;
                    namePreready = false;
                } else if (dlTagOpened && localName.equals("dd")) {
                    ddTagOpened = false;
                    if (companyReady) {
                        result.company = value;
                        companyReady = false;
                    } else if (typeReady) {
                        result.setType(value);
                        typeReady = false;
                    } else if (shapeReady) {
                        result.shape = value;
                        shapeReady = false;
                    } else if (effectReady) {
                        result.effect = value;
                        effectReady = false;
                    } else if (mainEffectReady) {
                        result.mainEffect = value;
                        mainEffectReady = false;
                    } else if (especiallyReady) {
                        result.especially = value;
                        especiallyReady = false;
                    } else if (elementReady) {
                        result.element = value.trim();
                        elementReady = false;
                    } else if (usageReady) {
                        result.usage = value;
                        usageReady = false;
                    } else if (parcelAndPriceReady) {
                        result.setParcelAndPrice(value);
                        parcelAndPriceReady = false;
                    } else if (carefulReady) {
                        result.careful = value;
                        carefulReady = false;
                    } else if (riskTypeReady) {
                        result.riskType = value;
                        riskTypeReady = false;
                    }
                } else if (sourceReady && localName.equals("p")) {
                    result.source = value.trim();
                    sourceReady = false;
                }
            }

            private StringBuilder sb = new StringBuilder();

            @Override
            public void characters(char[] ch, int start, int length) throws SAXException {
                if (dlTagOpened || sourceReady) {
                    sb.append(ch, start, length);
                }
            }
        };

        Html.ImageGetter imageGetter = null;
        Html.TagHandler tagHandler = new Html.TagHandler() {
            @Override
            public void handleTag(boolean opening, String tag, Editable output, XMLReader xmlReader) {
                if (tag.equals("body") && opening) {
                    xmlReader.setContentHandler(medicineDetailHandler);
                }
            }
        };

        Html.fromHtml(html.toString(), imageGetter, tagHandler);

        return result;
    }

    /**
     * HTMLを解析して、薬の詳細情報を取得する
     * 
     * @param html
     * @return MedicineDetailオブジェクト
     */
    public static final Medicine getMedicineDetailFromHtml(String html) {

        html = html.replace("<br>", "\n");

        // パターン1として解析
        Medicine result = parseHtmlAsPattern1(html);
        if (!result.validate()) {

            // パターン2として解析
            result = parseHtmlAsPattern2(html);
        }
        return result;
    }

    public static Bitmap getImage(String imageUrl) {
        Bitmap result = null;

        HttpGet entity = new HttpGet(imageUrl);

        try {
            DefaultHttpClient request = new DefaultHttpClient();
            HttpResponse res = request.execute(entity);
            result = BitmapFactory.decodeStream(res.getEntity().getContent());

        } catch (ClientProtocolException e) {
            Log.e(LOG_TAG, e.getMessage());
        } catch (IllegalStateException e) {
            Log.e(LOG_TAG, e.getMessage());
        } catch (IOException e) {
            Log.e(LOG_TAG, e.getMessage());
        }

        return result;
    }

}
